<?php
class User {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function register($email, $password, $firstName, $lastName) {
        try {
            // Vérifier si l'email existe déjà
            $stmt = $this->db->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->rowCount() > 0) {
                return [
                    'success' => false,
                    'message' => 'Cet email est déjà utilisé par un autre compte.'
                ];
            }
            
            // Hacher le mot de passe
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            
            // Insérer l'utilisateur
            $stmt = $this->db->prepare("
                INSERT INTO users (email, password, first_name, last_name, role, created_at)
                VALUES (?, ?, ?, ?, 'user', NOW())
            ");
            $result = $stmt->execute([$email, $hashedPassword, $firstName, $lastName]);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Votre compte a été créé avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la création de votre compte.'
                ];
            }
        } catch (PDOException $e) {
            return [
                'success' => false,
                'message' => 'Erreur de base de données: ' . $e->getMessage()
            ];
        }
    }

    public function login($email, $password) {
        try {
            $stmt = $this->db->prepare("
                SELECT id, email, password, first_name, last_name, role
                FROM users
                WHERE email = ?
            ");
            $stmt->execute([$email]);
            
            if ($stmt->rowCount() === 0) {
                return [
                    'success' => false,
                    'message' => 'Email ou mot de passe incorrect.'
                ];
            }
            
            $user = $stmt->fetch();
            
            if (password_verify($password, $user['password'])) {
                // Mettre à jour la date de dernière connexion
                $updateStmt = $this->db->prepare("
                    UPDATE users
                    SET last_login = NOW()
                    WHERE id = ?
                ");
                $updateStmt->execute([$user['id']]);
                
                // Supprimer le mot de passe du tableau avant de le retourner
                unset($user['password']);
                
                return [
                    'success' => true,
                    'message' => 'Connexion réussie.',
                    'user' => $user
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Email ou mot de passe incorrect.'
                ];
            }
        } catch (PDOException $e) {
            return [
                'success' => false,
                'message' => 'Erreur de base de données: ' . $e->getMessage()
            ];
        }
    }

    public function updateProfile($userId, $data) {
        try {
            // Récupérer les données actuelles de l'utilisateur
            $user = $this->getUserWithPassword($userId);
            
            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'Utilisateur non trouvé.'
                ];
            }
            
            // Si le mot de passe est fourni, vérifier et mettre à jour
            if (!empty($data['current_password']) && !empty($data['new_password'])) {
                // Vérifier le mot de passe actuel
                if (!password_verify($data['current_password'], $user['password'])) {
                    return [
                        'success' => false,
                        'message' => 'Le mot de passe actuel est incorrect.'
                    ];
                }
                
                // Mettre à jour le mot de passe
                $hashedPassword = password_hash($data['new_password'], PASSWORD_DEFAULT);
                $stmt = $this->db->prepare("UPDATE users SET password = ? WHERE id = ?");
                $stmt->execute([$hashedPassword, $userId]);
            }
            
            // Mettre à jour les autres informations
            $stmt = $this->db->prepare("
                UPDATE users
                SET first_name = ?, last_name = ?, email = ?
                WHERE id = ?
            ");
            $result = $stmt->execute([
                $data['first_name'] ?? $user['first_name'], 
                $data['last_name'] ?? $user['last_name'], 
                $data['email'] ?? $user['email'], 
                $userId
            ]);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Votre profil a été mis à jour avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la mise à jour de votre profil.'
                ];
            }
        } catch (PDOException $e) {
            return [
                'success' => false,
                'message' => 'Erreur de base de données: ' . $e->getMessage()
            ];
        }
    }

    private function getUserWithPassword($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT id, email, password, first_name, last_name, role
                FROM users
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            
            return $stmt->fetch();
        } catch (PDOException $e) {
            return false;
        }
    }

    public function getUserById($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT id, email, first_name, last_name, role, created_at, last_login
                FROM users
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            
            return $stmt->fetch();
        } catch (PDOException $e) {
            return false;
        }
    }

    public function getDiagnosticHistory($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT d.id, d.created_at
                FROM diagnostics d
                WHERE d.user_id = ?
                ORDER BY d.created_at DESC
            ");
            $stmt->execute([$userId]);
            
            $diagnostics = [];
            while ($row = $stmt->fetch()) {
                $diagnostics[] = $this->getDiagnosticSummary($row['id'], $userId);
            }
            
            return $diagnostics;
        } catch (PDOException $e) {
            return [];
        }
    }

    private function getDiagnosticSummary($diagnosticId, $userId) {
        try {
            // Obtenir les informations de base du diagnostic
            $stmt = $this->db->prepare("
                SELECT d.id, d.created_at
                FROM diagnostics d
                WHERE d.id = ? AND d.user_id = ?
            ");
            $stmt->execute([$diagnosticId, $userId]);
            $diagnostic = $stmt->fetch();
            
            if (!$diagnostic) {
                return null;
            }
            
            // Obtenir les symptômes du diagnostic
            $stmt = $this->db->prepare("
                SELECT s.name
                FROM diagnostic_symptom ds
                JOIN symptoms s ON ds.symptom_id = s.id
                WHERE ds.diagnostic_id = ?
            ");
            $stmt->execute([$diagnosticId]);
            $symptoms = [];
            while ($row = $stmt->fetch()) {
                $symptoms[] = $row['name'];
            }
            
            // Obtenir la maladie la plus probable
            $stmt = $this->db->prepare("
                SELECT d.name, dr.probability
                FROM diagnostic_results dr
                JOIN diseases d ON dr.disease_id = d.id
                WHERE dr.diagnostic_id = ?
                ORDER BY dr.probability DESC
                LIMIT 1
            ");
            $stmt->execute([$diagnosticId]);
            $topDisease = $stmt->fetch();
            
            return [
                'id' => $diagnostic['id'],
                'date' => $diagnostic['created_at'],
                'symptoms' => $symptoms,
                'diagnosis' => $topDisease ? $topDisease['name'] : 'Inconnu'
            ];
        } catch (PDOException $e) {
            return null;
        }
    }

    public function getUserByEmail($email) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM users WHERE email = ?");
            $stmt->execute([$email]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            return false;
        }
    }

    public function createUser($userData) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO users (email, password, first_name, last_name, role, created_at)
                VALUES (:email, :password, :first_name, :last_name, :role, :created_at)
            ");
            return $stmt->execute($userData);
        } catch (PDOException $e) {
            return false;
        }
    }

    public function updateUser($id, $updates) {
        try {
            $setClauses = [];
            $params = [];
            
            foreach ($updates as $key => $value) {
                $setClauses[] = "$key = :$key";
                $params[$key] = $value;
            }
            
            $params['id'] = $id;
            
            $sql = "UPDATE users SET " . implode(', ', $setClauses) . " WHERE id = :id";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            return false;
        }
    }

    public function updateLastLogin($id) {
        try {
            $stmt = $this->db->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
            return $stmt->execute([$id]);
        } catch (PDOException $e) {
            return false;
        }
    }

    public function getTotalUsers() {
        try {
            $stmt = $this->db->query("SELECT COUNT(*) FROM users");
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }

    public function countUsers() {
        return $this->getTotalUsers();
    }

    public function getRecentUsers($limit = 5) {
        try {
            $stmt = $this->db->prepare("SELECT id, first_name, last_name, email, created_at FROM users ORDER BY created_at DESC LIMIT :limit");
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
} 