<?php
class Disease {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getAllDiseases() {
        try {
            $stmt = $this->db->query("SELECT * FROM diseases ORDER BY name");
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error in getAllDiseases: " . $e->getMessage());
            return [];
        }
    }

    public function getDiseaseById($id) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM diseases WHERE id = ?");
            $stmt->execute([$id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error in getDiseaseById: " . $e->getMessage());
            return false;
        }
    }

    public function countDiseases() {
        try {
            $stmt = $this->db->query("SELECT COUNT(*) as count FROM diseases");
            $result = $stmt->fetch();
            return $result['count'];
        } catch (PDOException $e) {
            error_log("Error in countDiseases: " . $e->getMessage());
            return 0;
        }
    }

    public function addDisease($name, $description, $treatment = '') {
        try {
            // Vérifier si la maladie existe déjà
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM diseases WHERE name = ?");
            $stmt->execute([$name]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Une maladie avec ce nom existe déjà.'
                ];
            }
            
            $stmt = $this->db->prepare("INSERT INTO diseases (name, description, treatment) VALUES (?, ?, ?)");
            $success = $stmt->execute([$name, $description, $treatment]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'La maladie a été ajoutée avec succès.',
                    'id' => $this->db->lastInsertId()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de l\'ajout de la maladie.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in addDisease: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    public function updateDisease($id, $name, $description, $treatment = '') {
        try {
            // Vérifier si la maladie existe
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM diseases WHERE id = ?");
            $stmt->execute([$id]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] == 0) {
                return [
                    'success' => false,
                    'message' => 'La maladie n\'existe pas.'
                ];
            }
            
            // Vérifier si le nom existe déjà pour une autre maladie
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM diseases WHERE name = ? AND id != ?");
            $stmt->execute([$name, $id]);
            $nameExists = $stmt->fetch();
            
            if ($nameExists['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Une autre maladie avec ce nom existe déjà.'
                ];
            }
            
            $stmt = $this->db->prepare("UPDATE diseases SET name = ?, description = ?, treatment = ? WHERE id = ?");
            $success = $stmt->execute([$name, $description, $treatment, $id]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'La maladie a été mise à jour avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la mise à jour de la maladie.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in updateDisease: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    public function deleteDisease($id) {
        try {
            // Vérifier si la maladie existe
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM diseases WHERE id = ?");
            $stmt->execute([$id]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] == 0) {
                return [
                    'success' => false,
                    'message' => 'La maladie n\'existe pas.'
                ];
            }
            
            // Vérifier si la maladie est utilisée dans des relations
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM disease_symptom WHERE disease_id = ?");
            $stmt->execute([$id]);
            $hasRelations = $stmt->fetch();
            
            if ($hasRelations['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Cette maladie est utilisée dans des relations avec des symptômes. Veuillez d\'abord supprimer ces relations.'
                ];
            }
            
            $stmt = $this->db->prepare("DELETE FROM diseases WHERE id = ?");
            $success = $stmt->execute([$id]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'La maladie a été supprimée avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la suppression de la maladie.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in deleteDisease: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    // Gestion des relations entre maladies et symptômes
    public function getAllRelations() {
        try {
            $stmt = $this->db->query("
                SELECT r.id, r.disease_id, r.symptom_id, 
                d.name as disease_name, d.description as disease_description, 
                s.name as symptom_name, s.description as symptom_description, s.category as symptom_category
                FROM disease_symptom r
                JOIN diseases d ON r.disease_id = d.id
                JOIN symptoms s ON r.symptom_id = s.id
                ORDER BY d.name, s.name
            ");
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error in getAllRelations: " . $e->getMessage());
            return [];
        }
    }

    public function getRelationById($id) {
        try {
            $stmt = $this->db->prepare("
                SELECT r.*, d.name as disease_name, s.name as symptom_name
                FROM disease_symptom r
                JOIN diseases d ON r.disease_id = d.id
                JOIN symptoms s ON r.symptom_id = s.id
                WHERE r.id = ?
            ");
            $stmt->execute([$id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error in getRelationById: " . $e->getMessage());
            return false;
        }
    }

    public function addRelation($diseaseId, $symptomId, $probability = null) {
        try {
            // Vérifier si la relation existe déjà
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM disease_symptom WHERE disease_id = ? AND symptom_id = ?");
            $stmt->execute([$diseaseId, $symptomId]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Cette relation existe déjà.'
                ];
            }
            
            // Insérer la relation sans la probabilité
            $stmt = $this->db->prepare("INSERT INTO disease_symptom (disease_id, symptom_id) VALUES (?, ?)");
            $success = $stmt->execute([$diseaseId, $symptomId]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'La relation a été ajoutée avec succès.',
                    'id' => $this->db->lastInsertId()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de l\'ajout de la relation.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in addRelation: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    public function updateRelation($id, $diseaseId, $symptomId, $probability = null) {
        try {
            // Vérifier si la relation existe
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM disease_symptom WHERE id = ?");
            $stmt->execute([$id]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] == 0) {
                return [
                    'success' => false,
                    'message' => 'La relation n\'existe pas.'
                ];
            }
            
            // Vérifier si la nouvelle relation existe déjà pour un autre enregistrement
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM disease_symptom WHERE disease_id = ? AND symptom_id = ? AND id != ?");
            $stmt->execute([$diseaseId, $symptomId, $id]);
            $relationExists = $stmt->fetch();
            
            if ($relationExists['count'] > 0) {
                return [
                    'success' => false,
                    'message' => 'Une relation entre cette maladie et ce symptôme existe déjà.'
                ];
            }
            
            // Mise à jour de la relation sans le champ probability qui n'existe pas dans la table
            $stmt = $this->db->prepare("UPDATE disease_symptom SET disease_id = ?, symptom_id = ? WHERE id = ?");
            $success = $stmt->execute([$diseaseId, $symptomId, $id]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'La relation a été mise à jour avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la mise à jour de la relation.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in updateRelation: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }

    public function deleteRelation($id) {
        try {
            // Vérifier si la relation existe
            $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM disease_symptom WHERE id = ?");
            $stmt->execute([$id]);
            $exists = $stmt->fetch();
            
            if ($exists['count'] == 0) {
                return [
                    'success' => false,
                    'message' => 'La relation n\'existe pas.'
                ];
            }
            
            $stmt = $this->db->prepare("DELETE FROM disease_symptom WHERE id = ?");
            $success = $stmt->execute([$id]);
            
            if ($success) {
                return [
                    'success' => true,
                    'message' => 'La relation a été supprimée avec succès.'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Une erreur est survenue lors de la suppression de la relation.'
                ];
            }
        } catch (PDOException $e) {
            error_log("Error in deleteRelation: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Une erreur de base de données est survenue: ' . $e->getMessage()
            ];
        }
    }
} 