<?php
class Diagnostic {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getSymptoms(): array {
        try {
            $stmt = $this->db->query("
                SELECT * FROM symptoms 
                ORDER BY category, name
            ");
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    public function getDiseases() {
        try {
            $stmt = $this->db->query("
                SELECT * FROM diseases 
                ORDER BY name
            ");
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    public function getDiseaseSymptoms($diseaseId) {
        try {
            $stmt = $this->db->prepare("
                SELECT s.* 
                FROM symptoms s
                JOIN disease_symptom ds ON s.id = ds.symptom_id
                WHERE ds.disease_id = ?
            ");
            $stmt->execute([$diseaseId]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    public function analyzeSymptoms($selectedSymptoms) {
        try {
            if (empty($selectedSymptoms)) {
                return ['possible_diseases' => []];
            }

            // S'assurer que $selectedSymptoms est un tableau
            if (!is_array($selectedSymptoms)) {
                $selectedSymptoms = [$selectedSymptoms];
            }

            // Placeholders pour la requête préparée
            $placeholders = implode(',', array_fill(0, count($selectedSymptoms), '?'));
            
            // Récupérer les maladies qui correspondent aux symptômes sélectionnés
            $stmt = $this->db->prepare("
                SELECT d.id, d.name, d.description, 
                       COUNT(ds.symptom_id) as matching_symptoms,
                       (COUNT(ds.symptom_id) * 100.0 / (
                           SELECT COUNT(*) 
                           FROM disease_symptom 
                           WHERE disease_id = d.id
                       )) as probability
                FROM diseases d
                JOIN disease_symptom ds ON d.id = ds.disease_id
                WHERE ds.symptom_id IN ($placeholders)
                GROUP BY d.id, d.name
                ORDER BY probability DESC
            ");
            
            $stmt->execute($selectedSymptoms);
            $possibleDiseases = $stmt->fetchAll();
            
            // Formater les résultats
            $results = ['possible_diseases' => []];
            foreach ($possibleDiseases as $disease) {
                // Arrondir la probabilité à 2 décimales
                $probability = round(floatval($disease['probability']), 2);
                
                $results['possible_diseases'][] = [
                    'disease_id' => $disease['id'],
                    'name' => $disease['name'],
                    'description' => $disease['description'] ?? '',
                    'probability' => $probability
                ];
            }
            
            return $results;
        } catch (PDOException $e) {
            // En cas d'erreur, retourner un tableau vide
            return ['possible_diseases' => [], 'error' => $e->getMessage()];
        }
    }

    public function saveDiagnostic($userId, $symptoms, $results) {
        try {
            // S'assurer que $symptoms est un tableau
            if (!is_array($symptoms)) {
                $symptoms = [$symptoms];
            }
            
            $this->db->beginTransaction();
            
            // Insérer le diagnostic
            $stmt = $this->db->prepare("
                INSERT INTO diagnostics (user_id, created_at)
                VALUES (?, NOW())
            ");
            $stmt->execute([$userId]);
            $diagnosticId = $this->db->lastInsertId();
            
            // Insérer les symptômes du diagnostic
                $stmt = $this->db->prepare("
                    INSERT INTO diagnostic_symptom (diagnostic_id, symptom_id)
                    VALUES (?, ?)
                ");
            foreach ($symptoms as $symptomId) {
                $stmt->execute([$diagnosticId, $symptomId]);
            }
            
            // Insérer les résultats du diagnostic
            if (!empty($results) && !empty($results['possible_diseases'])) {
                    $stmt = $this->db->prepare("
                        INSERT INTO diagnostic_results (diagnostic_id, disease_id, probability)
                        VALUES (?, ?, ?)
                    ");
                foreach ($results['possible_diseases'] as $result) {
                    if (isset($result['disease_id']) && isset($result['probability'])) {
                        $stmt->execute([
                            $diagnosticId,
                            $result['disease_id'],
                            $result['probability']
                        ]);
                    }
                }
            }
            
            $this->db->commit();
            return true;
        } catch (PDOException $e) {
            $this->db->rollBack();
            return false;
        }
    }

    public function getDiagnosticById($id, $userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT d.*, d.created_at as date, u.email as user_email
                FROM diagnostics d
                JOIN users u ON d.user_id = u.id
                WHERE d.id = ? AND d.user_id = ?
            ");
            $stmt->execute([$id, $userId]);
            $diagnostic = $stmt->fetch();
            
            if ($diagnostic) {
                // Récupérer les symptômes
            $stmt = $this->db->prepare("
                SELECT s.name
                FROM diagnostic_symptom ds
                JOIN symptoms s ON ds.symptom_id = s.id
                WHERE ds.diagnostic_id = ?
            ");
                $stmt->execute([$id]);
                $diagnostic['symptoms'] = $stmt->fetchAll(PDO::FETCH_COLUMN);

                // Récupérer les résultats
            $stmt = $this->db->prepare("
                    SELECT d.id, d.name, d.description, dr.probability
                FROM diagnostic_results dr
                JOIN diseases d ON dr.disease_id = d.id
                WHERE dr.diagnostic_id = ?
                ORDER BY dr.probability DESC
            ");
                $stmt->execute([$id]);
                $diagnostic['possible_diseases'] = $stmt->fetchAll();
                
                // Ajouter des recommandations par défaut
                $diagnostic['recommendations'] = [
                    'Consultez un médecin pour un diagnostic précis',
                    'Reposez-vous et hydratez-vous correctement',
                    'Surveillez l\'évolution de vos symptômes'
                ];
                
                // Si nous avons des maladies possibles, ajouter des recommandations spécifiques
                if (!empty($diagnostic['possible_diseases'])) {
                    $topDiseaseId = $diagnostic['possible_diseases'][0]['id'];
                    $stmt = $this->db->prepare("
                        SELECT treatment_description 
                        FROM treatments 
                        WHERE disease_id = ?
                    ");
                    $stmt->execute([$topDiseaseId]);
                    $treatment = $stmt->fetch(PDO::FETCH_COLUMN);
                    
                    if ($treatment) {
                        $diagnostic['recommendations'][] = 'Traitement recommandé: ' . $treatment;
                    }
                }
            }

            return $diagnostic;
        } catch (PDOException $e) {
            return null;
        }
    }

    public function getUserDiagnostics($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT d.id, d.created_at,
                       (SELECT name 
                        FROM diseases di 
                        JOIN diagnostic_results dr ON di.id = dr.disease_id 
                        WHERE dr.diagnostic_id = d.id 
                        ORDER BY dr.probability DESC 
                        LIMIT 1) as top_disease,
                       (SELECT probability 
                        FROM diagnostic_results dr 
                        WHERE dr.diagnostic_id = d.id 
                        ORDER BY dr.probability DESC 
                        LIMIT 1) as top_probability,
                       GROUP_CONCAT(DISTINCT s.name SEPARATOR ', ') as symptoms
                FROM diagnostics d
                LEFT JOIN diagnostic_symptom ds ON d.id = ds.diagnostic_id
                LEFT JOIN symptoms s ON ds.symptom_id = s.id
                WHERE d.user_id = ?
                GROUP BY d.id, d.created_at
                ORDER BY d.created_at DESC
            ");
            $stmt->execute([$userId]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    public function searchUserDiagnostics($userId, $query) {
        try {
            $stmt = $this->db->prepare("
                SELECT DISTINCT d.id, d.created_at, 
                    GROUP_CONCAT(DISTINCT s.name SEPARATOR ', ') as symptoms,
                    (SELECT di.name 
                     FROM diseases di 
                     JOIN diagnostic_results dr ON di.id = dr.disease_id 
                     WHERE dr.diagnostic_id = d.id 
                     ORDER BY dr.probability DESC 
                     LIMIT 1) as top_disease,
                    (SELECT dr.probability 
                     FROM diagnostic_results dr 
                     WHERE dr.diagnostic_id = d.id 
                     ORDER BY dr.probability DESC 
                     LIMIT 1) as top_probability
                FROM diagnostics d
                JOIN diagnostic_symptom ds ON d.id = ds.diagnostic_id
                JOIN symptoms s ON ds.symptom_id = s.id
                WHERE d.user_id = ? AND 
                      (s.name LIKE ? OR 
                       EXISTS (SELECT 1 FROM diseases di 
                               JOIN diagnostic_results dr ON di.id = dr.disease_id 
                               WHERE dr.diagnostic_id = d.id AND di.name LIKE ?))
                GROUP BY d.id
                ORDER BY d.created_at DESC
            ");
            
            $searchTerm = "%{$query}%";
            $stmt->execute([$userId, $searchTerm, $searchTerm]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    // Méthodes pour l'administration des maladies et symptômes
    
    public function addDisease($name, $description, $treatment = '') {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO diseases (name, description, treatment, created_at)
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->execute([$name, $description, $treatment]);
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function updateDisease($id, $name, $description, $treatment = '') {
        try {
            $stmt = $this->db->prepare("
                UPDATE diseases
                SET name = ?, description = ?, treatment = ?
                WHERE id = ?
            ");
            return $stmt->execute([$name, $description, $treatment, $id]);
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function deleteDisease($id) {
        try {
            $stmt = $this->db->prepare("DELETE FROM diseases WHERE id = ?");
            return $stmt->execute([$id]);
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function addSymptom($name, $category, $description = '') {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO symptoms (name, category, description, created_at)
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->execute([$name, $category, $description]);
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function updateSymptom($id, $name, $category, $description = '') {
        try {
            $stmt = $this->db->prepare("
                UPDATE symptoms
                SET name = ?, category = ?, description = ?
                WHERE id = ?
            ");
            return $stmt->execute([$name, $category, $description, $id]);
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function deleteSymptom($id) {
        try {
            $stmt = $this->db->prepare("DELETE FROM symptoms WHERE id = ?");
            return $stmt->execute([$id]);
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function addDiseaseSymptomRelation($diseaseId, $symptomIds) {
        try {
            $this->db->beginTransaction();
            
            // Supprimer les relations existantes
            $stmt = $this->db->prepare("
                DELETE FROM disease_symptom
                WHERE disease_id = ?
            ");
            $stmt->execute([$diseaseId]);
            
            // Ajouter les nouvelles relations
            foreach ($symptomIds as $symptomId) {
                $stmt = $this->db->prepare("
                    INSERT INTO disease_symptom (disease_id, symptom_id)
                    VALUES (?, ?)
                ");
                $stmt->execute([$diseaseId, $symptomId]);
            }
            
            $this->db->commit();
            return true;
        } catch (PDOException $e) {
            $this->db->rollBack();
            return false;
        }
    }
    
    public function deleteDiseaseSymptomRelations($diseaseId) {
        try {
            $stmt = $this->db->prepare("
                DELETE FROM disease_symptom
                WHERE disease_id = ?
            ");
            return $stmt->execute([$diseaseId]);
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function getDiseaseSymptomRelations() {
        try {
            $stmt = $this->db->query("
                SELECT disease_id, GROUP_CONCAT(symptom_id) as symptom_ids
                FROM disease_symptom
                GROUP BY disease_id
            ");
            
            $relations = [];
            while ($row = $stmt->fetch()) {
                $relations[] = [
                    'disease_id' => $row['disease_id'],
                    'symptom_ids' => explode(',', $row['symptom_ids'])
                ];
            }
            
            return $relations;
        } catch (PDOException $e) {
            return [];
        }
    }
    
    public function getStats() {
        try {
            $stats = [];
            
            // Nombre total d'utilisateurs
            $stmt = $this->db->query("SELECT COUNT(*) as count FROM users");
            $stats['total_users'] = $stmt->fetch()['count'];
            
            // Nombre total de diagnostics
            $stmt = $this->db->query("SELECT COUNT(*) as count FROM diagnostics");
            $stats['total_diagnostics'] = $stmt->fetch()['count'];
            
            // Nombre total de maladies
            $stmt = $this->db->query("SELECT COUNT(*) as count FROM diseases");
            $stats['total_diseases'] = $stmt->fetch()['count'];
            
            // Nombre total de symptômes
            $stmt = $this->db->query("SELECT COUNT(*) as count FROM symptoms");
            $stats['total_symptoms'] = $stmt->fetch()['count'];
            
            return $stats;
        } catch (PDOException $e) {
            return [
                'total_users' => 0,
                'total_diagnostics' => 0,
                'total_diseases' => 0,
                'total_symptoms' => 0
            ];
        }
    }

    public function getTotalDiagnostics() {
        try {
            $stmt = $this->db->query("SELECT COUNT(*) FROM diagnostics");
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }

    public function countAllDiagnostics() {
        return $this->getTotalDiagnostics();
    }

    public function getRecentDiagnostics($limit = 5) {
        try {
            $stmt = $this->db->prepare("
                SELECT d.id, d.created_at, u.email as user_email,
                       (SELECT name 
                        FROM diseases di 
                        JOIN diagnostic_results dr ON di.id = dr.disease_id 
                        WHERE dr.diagnostic_id = d.id 
                        ORDER BY dr.probability DESC 
                        LIMIT 1) as top_disease
                FROM diagnostics d
                JOIN users u ON d.user_id = u.id
                ORDER BY d.created_at DESC
                LIMIT ?
            ");
            $stmt->execute([$limit]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    public function searchSymptoms($query) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM symptoms 
                WHERE name LIKE ? OR description LIKE ?
                ORDER BY category, name
            ");
            $searchTerm = "%{$query}%";
            $stmt->execute([$searchTerm, $searchTerm]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            return [];
        }
    }

    public function countUserDiagnostics($userId) {
        try {
            $stmt = $this->db->prepare("SELECT COUNT(*) FROM diagnostics WHERE user_id = ?");
            $stmt->execute([$userId]);
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }
} 